
// __________________________________________________________
//
//                          ntlib.c
//              Windows NT Utility Library V1.00
//                11-15-1998 Sven B. Schreiber
//                       sbs@orgon.com
// __________________________________________________________

#define  _NTLIB_DLL_
#include "NtLib.h"

// =================================================================
// DISCLAIMER
// =================================================================

/*

This software is provided "as is" and any express or implied
warranties, including, but not limited to, the implied warranties of
merchantibility and fitness for a particular purpose are disclaimed.
In no event shall the author Sven B. Schreiber be liable for any
direct, indirect, incidental, special, exemplary, or consequential
damages (including, but not limited to, procurement of substitute
goods or services; loss of use, data, or profits; or business
interruption) however caused and on any theory of liability,
whether in contract, strict liability, or tort (including negligence
or otherwise) arising in any way out of the use of this software,
even if advised of the possibility of such damage.

*/

// =================================================================
// REVISION HISTORY
// =================================================================

/*

11-15-1998 V1.00 Original version (SBS).

*/

// =================================================================
// GLOBAL VARIABLES
// =================================================================

SYSTEM_BASIC_INFORMATION gsbi;

// =================================================================
// GLOBAL STRINGS
// =================================================================

TBYTE atNull [] = T("");

// =================================================================
// LOOKUP TABLES
// =================================================================

NTL_LOOKUP aObjectTypes [] =
  {
  OB_TYPE_TYPE,                  T("Type"),
  OB_TYPE_DIRECTORY,             T("Directory"),
  OB_TYPE_SYMBOLIC_LINK,         T("SymbolicLink"),
  OB_TYPE_TOKEN,                 T("Token"),
  OB_TYPE_PROCESS,               T("Process"),
  OB_TYPE_THREAD,                T("Thread"),
  OB_TYPE_EVENT,                 T("Event"),
  OB_TYPE_EVENT_PAIR,            T("EventPair"),
  OB_TYPE_MUTANT,                T("Mutant"),
  OB_TYPE_SEMAPHORE,             T("Semaphore"),
  OB_TYPE_TIMER,                 T("Timer"),
  OB_TYPE_PROFILE,               T("Profile"),
  OB_TYPE_WINDOW_STATION,        T("WindowStation"),
  OB_TYPE_DESKTOP,               T("Desktop"),
  OB_TYPE_SECTION,               T("Section"),
  OB_TYPE_KEY,                   T("Key"),
  OB_TYPE_PORT,                  T("Port"),
  OB_TYPE_ADAPTER,               T("Adapter"),
  OB_TYPE_CONTROLLER,            T("Controller"),
  OB_TYPE_DEVICE,                T("Device"),
  OB_TYPE_DRIVER,                T("Driver"),
  OB_TYPE_IO_COMPLETION,         T("IoCompletion"),
  OB_TYPE_FILE,                  T("File"),
  };

#define N_OBJECT_TYPES (sizeof (aObjectTypes) / NTL_LOOKUP_)

// -----------------------------------------------------------------

NTL_LOOKUP aPoolTypes [] =
  {
  NonPagedPool,                  T("NonPagedPool"),
  PagedPool,                     T("PagedPool"),
  NonPagedPoolMustSucceed,       T("NonPagedPoolMustSucceed"),
  DontUseThisType,               T("DontUseThisType"),
  NonPagedPoolCacheAligned,      T("NonPagedPoolCacheAligned"),
  PagedPoolCacheAligned,         T("PagedPoolCacheAligned"),
  NonPagedPoolCacheAlignedMustS, T("NonPagedPoolCacheAlignedMustS"),
  };

#define N_POOL_TYPES (sizeof (aPoolTypes) / NTL_LOOKUP_)

// =================================================================
// USER INTERFACE
// =================================================================

int WINAPI
NtlBox (HWND   hWnd,
        UINT   uType,
        PTBYTE ptCaption,
        PTBYTE ptFormat,
        ...)
    {
    TBYTE atBuffer [NTL_WVSPRINTF];
    PVOID pArgs = &ptFormat + 1;

    wvsprintf (atBuffer, ptFormat, pArgs);
    return MessageBox (hWnd, atBuffer, ptCaption, uType);
    }

// =================================================================
// POINTER RETRIEVAL
// =================================================================

PVOID WINAPI
NtlPointerFs (DWORD dOffset)
    {
    DWORD dBackup;
    PVOID pData = NULL;

    __try
        {
        __asm
            {
            mov     dBackup, eax
            mov     eax, dOffset
            and     eax, 0xfffffffc
            mov     eax, fs:[eax]
            mov     pData, eax
            mov     eax, dBackup
            }
        }
    __except (EXCEPTION_EXECUTE_HANDLER)
        {
        pData = NULL;
        }
    return pData;
    }

// -----------------------------------------------------------------

PTEB WINAPI
NtlPointerTeb (void)
    {
    return NtCurrentTeb ();
    }

// -----------------------------------------------------------------

PPEB WINAPI
NtlPointerPeb (void)
    {
    PTEB pTeb = NtlPointerTeb ();

    return (pTeb != NULL ? pTeb->pPeb : NULL);
    }

// =================================================================
// DEVICE MANAGEMENT
// =================================================================

BOOL WINAPI
NtlDeviceClose (PHANDLE phDevice)
    {
    BOOL fOk = FALSE;

    if (*phDevice != INVALID_HANDLE_VALUE)
        {
        CloseHandle (*phDevice);
        *phDevice = INVALID_HANDLE_VALUE;

        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI
NtlDeviceOpen (PHANDLE phDevice,
               PTBYTE  atPath)
    {
    *phDevice = CreateFile (atPath, GENERIC_READ,
                            FILE_SHARE_READ | FILE_SHARE_WRITE,
                            NULL, OPEN_EXISTING,
                            FILE_ATTRIBUTE_NORMAL, NULL);

    return (*phDevice != INVALID_HANDLE_VALUE);
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlDeviceStop (PTBYTE atName,
               DWORD  dStatus)
    {
    SC_HANDLE      hControl, hService;
    SERVICE_STATUS ss;

    if ((dStatus & (NTL_STATUS_CREATED | NTL_STATUS_STARTED)) &&
        ((hControl = OpenSCManager (NULL, NULL,
                                    SC_MANAGER_ALL_ACCESS))
         != NULL))
        {
        if ((hService = OpenService (hControl, atName,
                                     SERVICE_ALL_ACCESS))
            != NULL)
            {
            if (dStatus & NTL_STATUS_STARTED)
                {
                ControlService (hService, SERVICE_CONTROL_STOP,
                                &ss);
                }
            if (dStatus & NTL_STATUS_CREATED)
                {
                DeleteService (hService);
                }
            CloseServiceHandle (hService);
            }
        CloseServiceHandle (hControl);
        }
    return 0;
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlDeviceStart (PTBYTE atName,
                PTBYTE atDisplay,
                PTBYTE atFile)
    {
    SC_HANDLE hControl, hService;
    DWORD     dStatus = 0;

    if ((hControl = OpenSCManager (NULL, NULL,
                                   SC_MANAGER_ALL_ACCESS))
        != NULL)
        {
        if ((hService = OpenService (hControl,
                                     atName,
                                     SERVICE_ALL_ACCESS))
            == NULL)
            {
            if ((hService = CreateService (hControl,
                                           atName, atDisplay,
                                           SERVICE_ALL_ACCESS,
                                           SERVICE_KERNEL_DRIVER,
                                           SERVICE_DEMAND_START,
                                           SERVICE_ERROR_NORMAL,
                                           atFile, NULL, NULL, NULL,
                                           NULL, NULL))
                != NULL)
                {
                dStatus |= NTL_STATUS_CREATED;
                }
            }
        if (hService != NULL)
            {
            if (StartService (hService, 0, NULL))
                {
                dStatus |= (NTL_STATUS_STARTED |
                            NTL_STATUS_RUNNING);
                }
            else
                {
                if (GetLastError ()
                    == ERROR_SERVICE_ALREADY_RUNNING)
                    {
                    dStatus |= NTL_STATUS_RUNNING;
                    }
                }
            CloseServiceHandle (hService);
            }
        CloseServiceHandle (hControl);
        }
    return dStatus;
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlDeviceCloseEx (PHANDLE phDevice,
                  PTBYTE  atName,
                  DWORD   dStatus)
    {
    if (dStatus & NTL_STATUS_OPENED)
        {
        NtlDeviceClose (phDevice);
        }
    return NtlDeviceStop (atName, dStatus);
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlDeviceOpenEx (PHANDLE phDevice,
                 PTBYTE  atName,
                 PTBYTE  atDisplay,
                 PTBYTE  atFile,
                 PTBYTE  atPath)
    {
    DWORD dStatus = 0;

    if (NtlDeviceOpen (phDevice, atPath))
        {
        dStatus |= NTL_STATUS_OPENED;
        }
    else
        {
        dStatus |= NtlDeviceStart (atName, atDisplay, atFile);

        if ((dStatus & NTL_STATUS_RUNNING) &&
            NtlDeviceOpen (phDevice, atPath))
            {
            dStatus |= NTL_STATUS_OPENED;
            }
        else
            {
            dStatus = NtlDeviceCloseEx (phDevice, atName, dStatus);
            }
        }
    return dStatus;
    }

// =================================================================
// COMMAND LINE PARSER
// =================================================================

PNTL_CMD_OPTION WINAPI
NtlCommandEnumerate (PNTL_CMD_OPTION pncoTable,
                     PTBYTE          ptName,
                     PDWORD          pdIndex)
    {
    DWORD           i, n;
    int             iSize;
    PNTL_CMD_OPTION pnco = NULL;

    for (n = 0; pncoTable [n].ptName != NULL; n++);

    for (i = *pdIndex; i < n; i++)
        {
        iSize = lstrlen (ptName);

        if ((pncoTable [i].Handler != NULL)
            &&
            (iSize <= lstrlen (pncoTable [i].ptName))
            &&
            (CompareString (LOCALE_USER_DEFAULT, NORM_IGNORECASE,
                            ptName, iSize,
                            pncoTable [i].ptName, iSize)
             == CSTR_EQUAL))
            {
            pnco = pncoTable + i;
            break;
            }
        }
    *pdIndex = (pnco != NULL ? ++i : 0);
    return pnco;
    }

// -----------------------------------------------------------------

PNTL_CMD_OPTION WINAPI
NtlCommandOption (PNTL_CMD_OPTION pncoTable,
                  PTBYTE          ptName,
                  PBOOL           pfAmbiguous)
    {
    int             iSize;
    PNTL_CMD_OPTION pnco, pncoNext;

    *pfAmbiguous = FALSE;
    pnco         = NULL;

    for (pncoNext = pncoTable; pncoNext->ptName != NULL; pncoNext++)
        {
        iSize = lstrlen (ptName);

        if ((pncoNext->Handler != NULL)
            &&
            (iSize <= lstrlen (pncoNext->ptName))
            &&
            (CompareString (LOCALE_USER_DEFAULT, NORM_IGNORECASE,
                            ptName, iSize,
                            pncoNext->ptName, iSize)
             == CSTR_EQUAL))
            {
            if (pnco == NULL)
                {
                pnco = pncoNext;
                }
            else
                {
                *pfAmbiguous = TRUE;
                pnco         = NULL;
                break;
                }
            }
        }
    return ((pnco != NULL) && (pnco->Handler != NULL) ? pnco
                                                      : NULL);
    }

// -----------------------------------------------------------------

void WINAPI
NtlCommandFlag (PNTL_CMD_FLAG pncfTable,
                PTBYTE        ptName,
                PDWORD        pdFlags)
    {
    DWORD i;

    for (i = 0; pncfTable [i].ptName != NULL; i++)
        {
        if (!lstrcmpi (ptName, pncfTable [i].ptName))
            {
            *pdFlags |= pncfTable [i].dMask;
            break;
            }
        }
    return;
    }

// -----------------------------------------------------------------

void WINAPI
NtlCommandParameter (PTBYTE             ptValue,
                     PNTL_CMD_PARAMETER pncp)
    {
    DWORD i, n;

    for (n = i = 0; ptValue [i]; i++)
        {
        if ((ptValue [i] < '0') || (ptValue [i] > '9')) break;

        n *= 10;
        n += (ptValue [i] - '0');
        }
    pncp->ptValue  = ptValue;
    pncp->dValue   = n;
    pncp->fNumeric = !ptValue [i];
    return;
    }

// -----------------------------------------------------------------

BOOL WINAPI
NtlCommandExecute (PNTL_CMD_DATA pncd)
    {
    BOOL fOk = FALSE;

    if ((pncd->pnco != NULL) && (pncd->pnco->Handler != NULL))
        {
        pncd->pnco->Handler (pncd);
        fOk = TRUE;
        }
    pncd->pnco   = NULL;
    pncd->dCount = 0;
    return fOk;
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlCommandEvaluate (PNTL_CMD_OPTION   pncoTable,
                    PNTL_CMD_FLAG     pncfTable,
                    PTBYTE            ptPrefixes,
                    PPTBYTE           pptData,
                    DWORD             dCount,
                    PNTL_CMD_CALLBACK CmdCallback,
                    PVOID             pClient)
    {
    PNTL_CMD_OPTION pnco;
    NTL_CMD_DATA    ncd;
    PTBYTE          ptName;
    BOOL            fAmbiguous, fMore;
    DWORD           i, j, k;
    DWORD           n = 0;

    ncd.pnco   = NULL;
    ncd.dCount = NTL_CMD_PARAMETERS;

    for (i = 0; i < dCount; i++)
        {
        for (j = 0; ptPrefixes [j]; j++)
            {
            if (ptPrefixes [j] == pptData [i] [0]) break;
            }
        if (ptPrefixes [j])
            {
            if (NtlCommandExecute (&ncd)) n++;

            if ((pnco = NtlCommandOption (pncoTable,
                                          pptData [i] + 1,
                                          &fAmbiguous))
                != NULL)
                {
                ncd.pnco      = pnco;
                ncd.pncfTable = pncfTable;
                ncd.tPrefix   = ptPrefixes [j];
                ncd.dFlags    = 0;
                ncd.dCount    = 0;
                ncd.dNumbers  = 0;
                }
            else
                {
                if (fAmbiguous)
                    {
                    fMore = CmdCallback (NTL_CMD_CALLBACK_AMBIGUOUS,
                                         ptPrefixes [j],
                                         pptData [i], pClient);

                    ptName = pptData [i] + 1;
                    }
                else
                    {
                    fMore = CmdCallback (NTL_CMD_CALLBACK_INVALID,
                                         ptPrefixes [j],
                                         pptData [i], pClient);

                    ptName = atNull;
                    }
                if (fMore)
                    {
                    k = 0;

                    while (((pnco = NtlCommandEnumerate (pncoTable,
                                                         ptName,
                                                         &k))
                            != NULL)
                           &&
                           CmdCallback (NTL_CMD_CALLBACK_ENUMERATE,
                                        ptPrefixes [j],
                                        pnco->ptName, pClient));

                    CmdCallback (NTL_CMD_CALLBACK_ENUMERATE,
                                 ptPrefixes [j],
                                 NULL, pClient);
                    }
                }
            }
        else
            {
            if (ncd.dCount < NTL_CMD_PARAMETERS)
                {
                NtlCommandFlag (pncfTable, pptData [i],
                                &ncd.dFlags);

                NtlCommandParameter (pptData [i],
                                     ncd.ancp + ncd.dCount);

                if (ncd.ancp [ncd.dCount++].fNumeric)
                    {
                    ncd.dNumbers++;
                    }
                }
            else
                {
                CmdCallback (NTL_CMD_CALLBACK_OVERFLOW,
                             ptPrefixes [j],
                             pptData [i], pClient);
                }
            }
        if ((i+1 == dCount) && NtlCommandExecute (&ncd)) n++;
        }
    return n;
    }

// =================================================================
// LINKED LIST MANAGEMENT
// =================================================================

PVOID WINAPI
NtlListNext (PLIST_ENTRY pleHead,
             DWORD       dOffset,
             PVOID       pCurrent)
    {
    PLIST_ENTRY ple;
    PVOID       pNext = NULL;

    ple = (pCurrent != NULL
           ? (PLIST_ENTRY) ((PBYTE) pCurrent + dOffset)
           : pleHead);

    if ((ple != NULL) && (ple->Flink != pleHead))
        {
        pNext = (PBYTE) ple->Flink - dOffset;
        }
    return pNext;
    }

// -----------------------------------------------------------------

PVOID WINAPI
NtlListFirst (PLIST_ENTRY pleHead,
              DWORD       dOffset)
    {
    return NtlListNext (pleHead, dOffset, NULL);
    }

// =================================================================
// BASIC TABLE MANAGEMENT
// =================================================================

NTSTATUS WINAPI
NtlTableReset (PNTL_TABLE pnt,
               PVOID      pData,
               DWORD      dData)
    {
    return NtlTableResetEx (pnt, pData, dData, 0, 0, 0, FALSE);
    }

// -----------------------------------------------------------------

NTSTATUS WINAPI
NtlTableResetEx (PNTL_TABLE pnt,
                 PVOID      pData,
                 DWORD      dData,
                 DWORD      dHeader,
                 DWORD      dEntry,
                 DWORD      dFlags,
                 BOOL       fCount)
    {
    NTSTATUS ns = STATUS_INVALID_PARAMETER;
    
    if (pnt != NULL)
        {
        pnt->pData   = pData;
        pnt->dData   = (pData != NULL ? dData : 0);
        pnt->dHeader = dHeader;
        pnt->dEntry  = dEntry;
        pnt->dFlags  = dFlags;
        pnt->dIndex  = 0;
        pnt->dCount  = (fCount ? NtlTableCount (pnt) : 0);

        ns = STATUS_SUCCESS;
        }
    return ns;
    }

// -----------------------------------------------------------------

NTSTATUS WINAPI
NtlTableLoad (PNTL_TABLE pnt,
              DWORD      dInfoGroup,
              DWORD      dInfoClass,
              HANDLE     hObject)
    {
    PVOID    pData, pData1;
    DWORD    dData, dData1;
    NTSTATUS ns = STATUS_INVALID_PARAMETER;

    if (pnt != NULL)
        {
        dData  = NTL_UNIT_TABLE;
        dData1 = 0;

        while ((pData = LocalAlloc (LMEM_FIXED, dData)) != NULL)
            {
            switch (dInfoGroup)
                {
                case NTL_INFO_SYSTEM:
                    {
                    ns = NtQuerySystemInformation
                             (dInfoClass,
                              pData, dData, &dData1);
                    break;
                    }
                case NTL_INFO_PROCESS:
                    {
                    ns = NtQueryInformationProcess
                             (hObject, dInfoClass,
                              pData, dData, &dData1);
                    break;
                    }
                case NTL_INFO_THREAD:
                    {
                    ns = NtQueryInformationThread
                             (hObject, dInfoClass,
                              pData, dData, &dData1);
                    break;
                    }
                case NTL_INFO_LOADER:
                    {
                    if (dInfoClass
                        == LoaderProcessModuleInformation)
                        {
                        ns = LdrQueryProcessModuleInformation
                                 (pData, dData, &dData1);
                        }
                    else
                        {
                        ns = STATUS_INVALID_INFO_CLASS;
                        }
                    break;
                    }
                }
            if (ns != STATUS_SUCCESS) dData1 = 0;
            if (ns != STATUS_INFO_LENGTH_MISMATCH) break;

            LocalFree (pData);
            dData += NTL_UNIT_TABLE;
            }
        dData = dData1;

        if (pData != NULL)
            {
            pData1 = pData;

            if ((ns == STATUS_SUCCESS)
                &&
                ((pData = LocalReAlloc (pData1, (dData ? dData : 1),
                                        LMEM_MOVEABLE))
                 == NULL))
                {
                ns = STATUS_NO_MEMORY;
                }
            if (ns != STATUS_SUCCESS)
                {
                LocalFree (pData1);

                pData = NULL;
                dData = 0;
                }
            }
        else
            {
            ns = STATUS_NO_MEMORY;
            }
        NtlTableReset (pnt, pData, dData);
        }
    return ns;
    }

// -----------------------------------------------------------------

NTSTATUS WINAPI
NtlTableLoadEx (PNTL_TABLE pnt,
                DWORD      dInfoGroup,
                DWORD      dInfoClass,
                HANDLE     hObject,
                DWORD      dHeader,
                DWORD      dEntry,
                DWORD      dFlags)
    {
    NTSTATUS ns = STATUS_INVALID_PARAMETER;
    
    if (pnt != NULL)
        {
        ns = NtlTableLoad (pnt, dInfoGroup, dInfoClass, hObject);

        NtlTableResetEx (pnt, pnt->pData, pnt->dData,
                         dHeader, dEntry, dFlags, TRUE);
        }
    return ns;
    }

// -----------------------------------------------------------------

NTSTATUS WINAPI
NtlTableUnload (PNTL_TABLE pnt)
    {
    NTSTATUS ns = STATUS_INVALID_PARAMETER;

    if (pnt != NULL)
        {
        if (pnt->pData != NULL)
            {
            ns = (LocalFree (pnt->pData) == NULL
                  ? STATUS_SUCCESS
                  : STATUS_UNSUCCESSFUL);
            }
        NtlTableReset (pnt, NULL, 0);
        }
    return ns;
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlTableSize (PNTL_TABLE pnt)
    {
    return ((pnt != NULL) && (pnt->pData != NULL) ? pnt->dData : 0);
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlTableCount (PNTL_TABLE pnt)
    {
    PVOID pData;
    DWORD dData, dOffset, dNext;
    DWORD n = 0;

    if (pnt != NULL)
        {
        if (pnt->pData != NULL)
            {
            if (pnt->dData >= pnt->dHeader + pnt->dEntry)
                {
                dData = pnt->dData - pnt->dHeader;

                if (pnt->dFlags & NTL_TABLE_COUNT)
                    {
                    if (pnt->dFlags & NTL_TABLE_FIXED)
                        {
                        n = (pnt->dEntry ? dData / pnt->dEntry : 1);
                        }
                    if (pnt->dHeader >= DWORD_)
                        {
                        n = min (n, *(PDWORD) pnt->pData);
                        }
                    }
                else
                    {
                    if (pnt->dFlags & NTL_TABLE_FIXED)
                        {
                        n = (pnt->dEntry ? dData / pnt->dEntry : 1);
                        }
                    else
                        {
                        dOffset = pnt->dHeader;

                        while (dOffset + max (pnt->dEntry, DWORD_)
                               <= pnt->dData)
                            {
                            pData = (PBYTE) pnt->pData + dOffset;
                            n++;

                            if (!(dNext = *(PDWORD) pData)) break;

                            if (pnt->dFlags & NTL_TABLE_ABSOLUTE)
                                {
                                dOffset  = dNext;
                                }
                            else
                                {
                                dOffset += dNext;
                                }
                            }
                        }
                    }
                }
            }
        pnt->dCount = n;
        }
    return n;
    }

// -----------------------------------------------------------------

PVOID WINAPI
NtlTableFirst (PNTL_TABLE pnt)
    {
    PVOID pData = NULL;

    if (pnt != NULL)
        {
        if (pnt->dCount)
            {
            pData = (PBYTE) pnt->pData + pnt->dHeader;
            }
        pnt->dIndex = 0;
        }
    return pData;
    }

// -----------------------------------------------------------------

PVOID WINAPI
NtlTableNext (PNTL_TABLE pnt,
              PVOID      pEntry)
    {
    DWORD dNext;
    PVOID pData = NULL;

    if (pnt != NULL)
        {
        if (pEntry == NULL)
            {
            pData = NtlTableFirst (pnt);
            }
        else
            {
            if (pnt->dIndex < pnt->dCount)
                {
                if (pnt->dFlags & NTL_TABLE_FIXED)
                    {
                    pData = (PBYTE) pEntry + pnt->dEntry;
                    }
                else
                    {
                    dNext = *(PDWORD) pEntry;

                    if (pnt->dFlags & NTL_TABLE_ABSOLUTE)
                        {
                        pData = (PBYTE) pnt->pData + dNext;
                        }
                    else
                        {
                        pData = (PBYTE) pEntry     + dNext;
                        }
                    }
                pnt->dIndex++;
                }
            }
        }
    return pData;
    }

// =================================================================
// STANDARD TABLES
// =================================================================

NTSTATUS WINAPI
NtlTableCpu (PNTL_TABLE pnt)
    {
    return NtlTableLoadEx (pnt,
                           NTL_INFO_SYSTEM,
                           SystemProcessorCounters,
                           NULL,
                           NTL_CPU_HEADER,
                           NTL_CPU_ENTRY,
                           NTL_CPU_ATTRIBUTES);
    }

// -----------------------------------------------------------------

NTSTATUS WINAPI
NtlTableHandle (PNTL_TABLE pnt)
    {
    return NtlTableLoadEx (pnt,
                           NTL_INFO_SYSTEM,
                           SystemHandleInformation,
                           NULL,
                           NTL_HANDLE_HEADER,
                           NTL_HANDLE_ENTRY,
                           NTL_HANDLE_ATTRIBUTES);
    }

// -----------------------------------------------------------------

NTSTATUS WINAPI
NtlTableLock (PNTL_TABLE pnt)
    {
    return NtlTableLoadEx (pnt,
                           NTL_INFO_SYSTEM,
                           SystemLockInformation,
                           NULL,
                           NTL_LOCK_HEADER,
                           NTL_LOCK_ENTRY,
                           NTL_LOCK_ATTRIBUTES);
    }

// -----------------------------------------------------------------

NTSTATUS WINAPI
NtlTableLookaside (PNTL_TABLE pnt)
    {
    return NtlTableLoadEx (pnt,
                           NTL_INFO_SYSTEM,
                           SystemLookasideInformation,
                           NULL,
                           NTL_LOOKASIDE_HEADER,
                           NTL_LOOKASIDE_ENTRY,
                           NTL_LOOKASIDE_ATTRIBUTES);
    }

// -----------------------------------------------------------------

NTSTATUS WINAPI
NtlTableModule (PNTL_TABLE pnt,
                BOOL       fProcess)
    {
    return NtlTableLoadEx (pnt,
                           (fProcess
                            ? NTL_INFO_LOADER
                            : NTL_INFO_SYSTEM),
                           (fProcess
                            ? LoaderProcessModuleInformation
                            : SystemModuleInformation),
                           NULL,
                           NTL_MODULE_HEADER,
                           NTL_MODULE_ENTRY,
                           NTL_MODULE_ATTRIBUTES);
    }

// -----------------------------------------------------------------

NTSTATUS WINAPI
NtlTablePageFile (PNTL_TABLE pnt)
    {
    return NtlTableLoadEx (pnt,
                           NTL_INFO_SYSTEM,
                           SystemPageFileInformation,
                           NULL,
                           NTL_PAGE_FILE_HEADER,
                           NTL_PAGE_FILE_ENTRY,
                           NTL_PAGE_FILE_ATTRIBUTES);
    }

// -----------------------------------------------------------------

NTSTATUS WINAPI
NtlTableObject (PNTL_TABLE pnt)
    {
    return NtlTableLoadEx (pnt,
                           NTL_INFO_SYSTEM,
                           SystemObjectInformation,
                           NULL,
                           NTL_OBJECT_HEADER,
                           NTL_OBJECT_ENTRY,
                           NTL_OBJECT_ATTRIBUTES);
    }

// -----------------------------------------------------------------

NTSTATUS WINAPI
NtlTableProcess (PNTL_TABLE pnt)
    {
    return NtlTableLoadEx (pnt,
                           NTL_INFO_SYSTEM,
                           SystemProcessInformation,
                           NULL,
                           NTL_PROCESS_HEADER,
                           NTL_PROCESS_ENTRY,
                           NTL_PROCESS_ATTRIBUTES);
    }

// =================================================================
// STANDARD SUBTABLES
// =================================================================

PSYSTEM_OBJECT WINAPI
NtlTableObjectFirst (PNTL_TABLE                 pntSub,
                     PNTL_TABLE                 pnt,
                     PSYSTEM_OBJECT_INFORMATION psoi)
    {
    DWORD dHeader;

    dHeader = ((PBYTE) psoi->usTypeName.Buffer
                     + psoi->usTypeName.MaximumLength)
              - (PBYTE) psoi;

    NtlTableResetEx (pntSub, pnt->pData, pnt->dData,
                     dHeader, SYSTEM_OBJECT_, NTL_TABLE_ABSOLUTE,
                     TRUE);

    return NtlTableFirst (pntSub);
    }

// -----------------------------------------------------------------

PSYSTEM_OBJECT WINAPI
NtlTableObjectNext (PNTL_TABLE     pntSub,
                    PSYSTEM_OBJECT pso)
    {
    return NtlTableNext (pntSub, pso);
    }

// =================================================================
// FIXED SIZE INFO MANAGEMENT
// =================================================================

NTSTATUS WINAPI
NtlInfoRead (PVOID  pData,
             PDWORD pdData,
             DWORD  dInfoGroup,
             DWORD  dInfoClass,
             HANDLE hObject)
    {
    DWORD    dData = 0;
    NTSTATUS ns    = STATUS_INVALID_PARAMETER;

    if (pdData != NULL)
        {
        if ((pData != NULL) && *pdData)
            {
            switch (dInfoGroup)
                {
                case NTL_INFO_SYSTEM:
                    {
                    ns = NtQuerySystemInformation
                             (dInfoClass,
                              pData, *pdData, &dData);
                    break;
                    }
                case NTL_INFO_PROCESS:
                    {
                    ns = NtQueryInformationProcess
                             (hObject, dInfoClass,
                              pData, *pdData, &dData);
                    break;
                    }
                case NTL_INFO_THREAD:
                    {
                    ns = NtQueryInformationThread
                             (hObject, dInfoClass,
                              pData, *pdData, &dData);
                    break;
                    }
                case NTL_INFO_LOADER:
                    {
                    if (dInfoClass
                        == LoaderProcessModuleInformation)
                        {
                        ns = LdrQueryProcessModuleInformation
                                 (pData, *pdData, &dData);
                        }
                    else
                        {
                        ns = STATUS_INVALID_INFO_CLASS;
                        }
                    break;
                    }
                }
            }
        *pdData = dData;
        }
    return ns;
    }

// =================================================================
// MEMORY SIZE CONVERSION ROUTINES
// =================================================================

DWORD WINAPI
NtlBytesToKb (DWORD dBytes)
    {
    return (dBytes ? ((dBytes-1) >> 10) + 1 : 0);
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlBytesToMb (DWORD dBytes)
    {
    return (dBytes ? ((dBytes-1) >> 20) + 1 : 0);
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlBytesToGb (DWORD dBytes)
    {
    return (dBytes ? ((dBytes-1) >> 30) + 1 : 0);
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlBytesToPages (DWORD dBytes)
    {
    return (dBytes ? ((dBytes-1) / gsbi.dPageSize) + 1 : 0);
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlPagesToBytes (DWORD dPages)
    {
    return (dPages * gsbi.dPageSize);
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlPagesToKb (DWORD dPages)
    {
    return NtlBytesToKb (NtlPagesToBytes (dPages));
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlPagesToMb (DWORD dPages)
    {
    return NtlBytesToMb (NtlPagesToBytes (dPages));
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlPagesToGb (DWORD dPages)
    {
    return NtlBytesToGb (NtlPagesToBytes (dPages));
    }

// =================================================================
// TIME CONVERSION ROUTINES
// =================================================================

void WINAPI
NtlTimeReset (PQWORD pqTime)
    {
    pqTime->LowPart = pqTime->HighPart = 0;
    return;
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlTimeUnits (PQWORD pqTime,
              DWORD  dUnit)
    {
    BOOL  fNegative;
    DWORD dRemainder = 0;

    if (dUnit > 1)
        {
        fNegative = RtlLargeIntegerLessThanZero (*pqTime);

        if (fNegative) *pqTime = RtlLargeIntegerNegate (*pqTime);

        *pqTime = RtlExtendedLargeIntegerDivide (*pqTime, dUnit,
                                                 &dRemainder);

        if (fNegative) *pqTime = RtlLargeIntegerNegate (*pqTime);
        }
    return dRemainder;
    }

// -----------------------------------------------------------------

DWORD WINAPI
NtlTimeInterval (PQWORD pqTime1,
                 PQWORD pqTime2,
                 PQWORD pqInterval,
                 DWORD  dUnit)
    {
    *pqInterval = RtlLargeIntegerSubtract (*pqTime2, *pqTime1);
    return NtlTimeUnits (pqInterval, dUnit);
    }

// -----------------------------------------------------------------

void WINAPI
NtlTimeExtend (PQWORD pqTime,
               PQWORD pqInterval,
               DWORD  dUnit)
    {
    QWORD qInterval;

    qInterval = (dUnit > 1
                 ? RtlExtendedIntegerMultiply (*pqInterval, dUnit)
                 : *pqInterval);

    *pqTime = RtlLargeIntegerAdd (*pqTime, qInterval);
    return;
    }

// -----------------------------------------------------------------

BOOL WINAPI
NtlTimePack (PTIME_FIELDS ptfTime,
             PQWORD       pqTime)
    {
    return (RtlTimeFieldsToTime (ptfTime, pqTime) != FALSE);
    }

// -----------------------------------------------------------------

void WINAPI
NtlTimeUnpack (PQWORD       pqTime,
               PTIME_FIELDS ptfTime)
    {
    RtlTimeToTimeFields (pqTime, ptfTime);
    return;
    }

// =================================================================
// ID TO NAME CONVERSION
// =================================================================

PTBYTE WINAPI
NtlNameLookup (PNTL_LOOKUP pnl,
               DWORD       dCount,
               DWORD       dCode,
               PDWORD      pdSize,
               PDWORD      pdSizeMax)
    {
    DWORD  dSizeMax, i, n;
    PTBYTE ptName = atNull;

    dSizeMax = 0;

    for (i = 0; i < dCount; i++)
        {
        if (pnl [i].dCode == dCode) ptName = pnl [i].ptName;
        n = lstrlen (pnl [i].ptName);
        dSizeMax = max (dSizeMax, n);
        }
    if (pdSize    != NULL) *pdSize    = lstrlen (ptName);
    if (pdSizeMax != NULL) *pdSizeMax = dSizeMax;
    return ptName;
    }

// -----------------------------------------------------------------

PTBYTE WINAPI
NtlNameObjectType (DWORD  dType,
                   PDWORD pdSize,
                   PDWORD pdSizeMax)
    {
    return NtlNameLookup (aObjectTypes, N_OBJECT_TYPES,
                          dType, pdSize, pdSizeMax);
    }

// -----------------------------------------------------------------

PTBYTE WINAPI
NtlNamePoolType (DWORD  dType,
                 PDWORD pdSize,
                 PDWORD pdSizeMax)
    {
    return NtlNameLookup (aPoolTypes, N_POOL_TYPES,
                          dType, pdSize, pdSizeMax);
    }

// =================================================================
// DLL MANAGEMENT
// =================================================================

BOOL WINAPI
DllMain (HINSTANCE hInstance,
         DWORD     dReason,
         PVOID     pReserved)
    {
    BOOL fOk = TRUE;

    if (dReason == DLL_PROCESS_ATTACH)
        {
        fOk = (NtQuerySystemInformation (SystemBasicInformation,
                                         &gsbi,
                                         SYSTEM_BASIC_INFORMATION_,
                                         NULL)
               == STATUS_SUCCESS);
        }
    return fOk;
    }

// =================================================================
// END OF PROGRAM
// =================================================================
